import { defineStore } from 'pinia';

const useCartStore = defineStore("cart", {
  state: () => ({
    items: []
  }),
  getters: {
    totalItems: (state) => {
      return state.items.reduce((sum, item) => sum + item.quantity, 0);
    },
    totalPrice: (state) => {
      return state.items.reduce((sum, item) => {
        const price = item.sale_price || item.price;
        return sum + price * item.quantity;
      }, 0);
    },
    getItemQuantity: (state) => {
      return (productId, variationId) => {
        const item = state.items.find(
          (item2) => item2.product_id === productId && item2.variation_id === variationId
        );
        return item ? item.quantity : 0;
      };
    }
  },
  actions: {
    increaseQuantity(productId, variationId) {
      const item = this.items.find(
        (i) => i.product_id === productId && i.variation_id === variationId
      );
      if (item && item.quantity < item.max_quantity) {
        item.quantity++;
      } else if (!item) {
        console.warn("محصول در سبد خرید یافت نشد");
      }
    },
    decreaseQuantity(productId, variationId) {
      const item = this.items.find(
        (i) => i.product_id === productId && i.variation_id === variationId
      );
      if (item) {
        if (item.quantity > 1) {
          item.quantity--;
        } else {
          this.removeItem(productId, variationId);
        }
      }
    },
    addItem(item) {
      const existing = this.items.find(
        (i) => i.product_id === item.product_id && i.variation_id === item.variation_id
      );
      if (existing) {
        const newQuantity = Math.min(
          existing.quantity + item.quantity,
          item.max_quantity
        );
        existing.quantity = newQuantity;
      } else {
        this.items.push(item);
      }
    },
    removeItem(productId, variationId) {
      this.items = this.items.filter(
        (i) => !(i.product_id === productId && i.variation_id === variationId)
      );
    },
    clearCart() {
      this.items = [];
    },
    // updateItemPrice(productId, variationId, newPrice) {
    //   const item = this.items.find(
    //     (i) => i.product_id === productId && i.variation_id === variationId
    //   );
    //   if (!item) return;
    //   if (item.sale_price) {
    //     item.sale_price = newPrice;
    //   } else {
    //     item.price = newPrice;
    //   }
    // },
    applyServerChanges(changes) {
      changes.forEach((change) => {
        const item = this.items.find(
          (i) => i.variation_id === change.variation_id
        );
        if (!item) return;
        if (change.priceChanged) {
          item.price = change.price;
          item.sale_price = change.sale_price;
          item.discount_percent = change.discount_percent;
        }
        if (change.quantityChanged) {
          item.max_quantity = change.quantity;
          if (item.quantity > change.quantity) {
            item.quantity = change.quantity;
          }
        }
      });
    }
  },
  persist: {
    storage: void 0,
    pick: ["items"]
  }
});

export { useCartStore as u };
//# sourceMappingURL=cart-BfofACZK.mjs.map
